// ============================================================================
//
// = LIBRARY
//     PCIECONFIGSPACE
//
// = FILENAME
//     PCIEMemoryMgmt.cpp
//
// = COPYRIGHT
//     (C) Copyright 2007 Agilent Technologies
//
// ============================================================================

#include "AgtPCIPort.h"
#include "devpciedefs.h"
#include "devpcieintdefs.h"
#include "PCIEMemoryMgmt.h"
#include "PCIEFPGAFramework.h"

//-----------------------------------------------------------------------------
CPCIEMemoryMgmt::CPCIEMemoryMgmt() 
{
  UInt8 i;

  for( i = 0; i < ( UInt8 )PCIEMEMORYMGMT_NOFHWCHANNELS; i++ )
  {
    mReqBehGenEnabled[i] = 1;
  }

  for( i = 0; i < ( UInt8 )PCIEMEMORYMGMT_NOFQUEUES; i++ )
  {
    mCompBehGenEnabled[i] = 1;
  }

  for( UInt32 ii = 0; ii < PCIE_MAXIMUM_NUMBER_OF_PROBE_BOARDS; ii++ )
  {
    for( i = 0; i < ( UInt8 )PCIEMEMORYMGMT_NOFHWCHANNELS; i++ )
    {
      mSingleShotEnabled[ii][i] = 0x1;
      mBlockLineCount[ii][i] = 0x0;
      mReqBehLineCount[ii][i] = 0x0;

      if( ( EPCIEHwChannelFunction )i == PCIE_HWCHANNEL_FUNCTION_A )
      {
        mBlockGenEnabled[ii][i] = 0x1;
      }
      else
      {
        mBlockGenEnabled[ii][i] = 0x0;
      }
    }

    for( i = 0; i < ( UInt8 )PCIEMEMORYMGMT_NOFQUEUES; i++ )
    {
      mCompBehLineCount[ii][i] = 0x0;
    }
  }

  // Default for all packet types
  for( i = 0; i < ( UInt8 )MEMORYMGMT_BLOCKLINETYPE_NOFENTRIES; i++ )
  {
    for( UInt8 j = 0; j < ( UInt8 )PCIE_MAX_BLOCKPROP_NUM; j++ )
    {
      switch( j )
      {
      case PCIE_BLOCK_LEN:
      case PCIE_BLOCK_RSVD023:
      case PCIE_BLOCK_ATTR:
      case PCIE_BLOCK_RSVD013:
      case PCIE_BLOCK_TC:
      case PCIE_BLOCK_RSVD017:
      case PCIE_BLOCK_TYPE:
      case PCIE_BLOCK_FMT:
      case PCIE_BLOCK_RSVD007:
      case PCIE_BLOCK_WAITREPEAT:
      case PCIE_BLOCK_PATTERN_TERM:
        mBlockLineValid[i].mBlock[j] = 1; // Property not dependent of packet type
        break;
      default:
        mBlockLineValid[i].mBlock[j] = 0; // Property dependent of packet type
      }
    }
  }

  // mem64 line
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_MEM64].mBlock[PCIE_BLOCK_1STDWBE] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_MEM64].mBlock[PCIE_BLOCK_LASTDWBE] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_MEM64].mBlock[PCIE_BLOCK_MEM64_ADDRHI] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_MEM64].mBlock[PCIE_BLOCK_MEM64_ADDRLO] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_MEM64].mBlock[PCIE_BLOCK_MEM64_RSVD151] = 1;

  // mem32 line
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_MEM32].mBlock[PCIE_BLOCK_1STDWBE] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_MEM32].mBlock[PCIE_BLOCK_LASTDWBE] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_MEM32].mBlock[PCIE_BLOCK_MEM32_ADDR] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_MEM32].mBlock[PCIE_BLOCK_MEM32_RSVD111] = 1;

  // io line
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_IO].mBlock[PCIE_BLOCK_1STDWBE] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_IO].mBlock[PCIE_BLOCK_LASTDWBE] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_IO].mBlock[PCIE_BLOCK_IO_ADDR] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_IO].mBlock[PCIE_BLOCK_IO_RSVD111] = 1;

  // msg as line
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_MSGAS].mBlock[PCIE_BLOCK_MSGAS_RSVD047] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_MSGAS].mBlock[PCIE_BLOCK_MSGAS_RSVD087] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_MSGAS].mBlock[PCIE_BLOCK_MSGAS_RSVD127] = 1;

  // config transaction line
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_CFG].mBlock[PCIE_BLOCK_1STDWBE] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_CFG].mBlock[PCIE_BLOCK_LASTDWBE] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_CFG].mBlock[PCIE_BLOCK_CFG_RSVD111] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_CFG].mBlock[PCIE_BLOCK_CFG_REGNUM] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_CFG].mBlock[PCIE_BLOCK_CFG_EXTREGNUM] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_CFG].mBlock[PCIE_BLOCK_CFG_RSVD107] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_CFG].mBlock[PCIE_BLOCK_CFG_FUNCNUM] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_CFG].mBlock[PCIE_BLOCK_CFG_DEVNUM] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_CFG].mBlock[PCIE_BLOCK_CFG_BUSNUM] = 1;

  // message request header
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_MSG].mBlock[PCIE_BLOCK_MCODE] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_MSG].mBlock[PCIE_BLOCK_MSG_BYTES08_11] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_MSG].mBlock[PCIE_BLOCK_MSG_BYTES12_15] = 1;

  // msgasd line
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_MSGASD].mBlock[PCIE_BLOCK_MSGASD_RSVD047] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_MSGASD].mBlock[PCIE_BLOCK_MSGASD_RSVD087] = 1;
  mBlockLineValid[MEMORYMGMT_BLOCKLINETYPE_MSGASD].mBlock[PCIE_BLOCK_MSGASD_RSVD127] = 1;

  for( UInt32 ii = 0; ii < PCIE_MAXIMUM_NUMBER_OF_PROBE_BOARDS; ii++ )
  {
    mBlockLineCount[ii][0] = 0x1;
    mReqBehLineCount[ii][0] = 0x1;
    mCompBehLineCount[ii][0] = 0x1;
  }
}

//-----------------------------------------------------------------------------
CPCIEMemoryMgmt::~CPCIEMemoryMgmt()
{
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::BlockSet
(
  const AgtPortHandleT& portHandle,
  const EPCIEHwChannelFunction& channel, 
  const UInt32& line, 
  const EPCIEBlock& prop,
  const UInt32& val
)
{
  CPCIEFpgaFramework::instance().getFpgaTxMemories().setBlockProp( portHandle, prop, channel, line, val );
  
  if(prop == PCIE_BLOCK_REPEAT)
  {
    updateRepeatFirst_Block( portHandle, channel );
  }
}

// ----------------------------------------------------------------------------
UInt32 CPCIEMemoryMgmt::BlockGet
(
 const AgtPortHandleT& portHandle, 
 const EPCIEHwChannelFunction& channel, 
 const UInt32& line, 
 const EPCIEBlock& prop
 )
{
  return CPCIEFpgaFramework::instance().getFpgaTxMemories().getBlockProp(portHandle, prop,channel, line);
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::BlockLineWrite
(
 const AgtPortHandleT& portHandle, 
 const EPCIEHwChannelFunction& channel, 
 const UInt32& lineNum, 
 const UInt32& numBytes, 
 const UInt8* mData
 )
{
  LineWrite(portHandle, MEMORYMGMT_LINETYPE_BLOCK,(UInt8)channel,lineNum,numBytes,mData);
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::BlockLineRead
(
 const AgtPortHandleT& portHandle, 
 const EPCIEHwChannelFunction& channel, 
 const UInt32& lineNum, 
 const UInt32& numBytes, 
 UInt8* theData
 )
{
  LineRead(portHandle, MEMORYMGMT_LINETYPE_BLOCK,(UInt8)channel,lineNum,numBytes,theData);
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::BlockLineDefaultSet
(
 const AgtPortHandleT& portHandle, 
 const EPCIEHwChannelFunction& channel, 
 const UInt32& lineNum
 )
{
  CPCIEFpgaFramework::instance().getFpgaTxMemories().setBlockLineDefault(portHandle, channel, lineNum);
  
  updateRepeatFirst_Block( portHandle, channel );
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::BlockLineDefaultInsert
(
 const AgtPortHandleT& portHandle, 
 const EPCIEHwChannelFunction& channel, 
 const UInt32& lineNum
 )
{
  // it's not possible to insert a line if block mem is full !
  if(lineNum < PCIE_BLOCKLINE_NUM && mBlockLineCount[portHandle][(UInt8)channel] < PCIE_BLOCKLINE_NUM)
  {
    if( CPCIEFpgaFramework::instance().getFpgaTxMemories().insertBlockLine( portHandle, channel, lineNum,
      mBlockLineCount[portHandle][( UInt8 )channel] ) )
    {          
      mBlockLineCount[portHandle][(UInt8)channel]++;
      BlockLineDefaultSet(portHandle, channel,lineNum);
    }
  }
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::BlockLineDelete
(
 const AgtPortHandleT& portHandle, 
 const EPCIEHwChannelFunction& channel, 
 const UInt32& lineNum
 )
{
  if(lineNum < PCIE_BLOCKLINE_NUM && mBlockLineCount[portHandle][(UInt8)channel] > 0)
  {
    if( CPCIEFpgaFramework::instance().getFpgaTxMemories().removeBlockLine( portHandle, channel, lineNum,
      mBlockLineCount[portHandle][( UInt8 )channel] ) )
    {     
      mBlockLineCount[portHandle][(UInt8)channel]--;
      BlockLineDefaultSet(portHandle, channel,mBlockLineCount[portHandle][(UInt8)channel]);   
    }
  }
}

// ----------------------------------------------------------------------------
UInt32 CPCIEMemoryMgmt::BlockLineCountGet
(
  const AgtPortHandleT& portHandle,
  const EPCIEHwChannelFunction& channel
)
{
  return mBlockLineCount[portHandle][(UInt8)channel];
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::ReqBehSet
(
 const AgtPortHandleT& portHandle,
 const EPCIEHwChannelFunction& channel, 
 const UInt32& line, 
 const EPCIEReqBeh& prop,
 const UInt32& val
 )
{
  CPCIEFpgaFramework::instance().getFpgaTxMemories().setReqBehProp(portHandle,prop,channel, line, val);

  if(prop == PCIE_REQBEH_REPEAT)
  {
    updateRepeatFirst_ReqBeh( portHandle, channel );
  }
}

// ----------------------------------------------------------------------------
UInt32 CPCIEMemoryMgmt::ReqBehGet
(
 const AgtPortHandleT& portHandle, 
 const EPCIEHwChannelFunction& channel, 
 const UInt32& line, 
 const EPCIEReqBeh& prop
 )
{
  return CPCIEFpgaFramework::instance().getFpgaTxMemories().getReqBehProp(portHandle,prop,channel, line);
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::ReqBehLineWrite
(
 const AgtPortHandleT& portHandle, 
 const EPCIEHwChannelFunction& channel, 
 const UInt32& lineNum, 
 const UInt32& numBytes, 
 const UInt8* mData
 )
{
  LineWrite(portHandle, MEMORYMGMT_LINETYPE_REQBEH,(UInt8)channel,lineNum,numBytes,mData);
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::ReqBehLineRead
(
 const AgtPortHandleT& portHandle, 
 const EPCIEHwChannelFunction& channel, 
 const UInt32& lineNum, 
 const UInt32& numBytes, 
 UInt8* theData
 )
{
  LineRead(portHandle, MEMORYMGMT_LINETYPE_REQBEH,(UInt8)channel,lineNum,numBytes,theData);
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::ReqBehLineDefaultSet
(
 const AgtPortHandleT& portHandle, 
 const EPCIEHwChannelFunction& channel, 
 const UInt32& lineNum
 )
{
  CPCIEFpgaFramework::instance().getFpgaTxMemories().setReqBehLineDefault(portHandle,channel, lineNum);

  updateRepeatFirst_ReqBeh( portHandle, channel );
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::ReqBehLineDefaultInsert
(
 const AgtPortHandleT& portHandle, 
 const EPCIEHwChannelFunction& channel, 
 const UInt32& lineNum
 )
{
  // it's not possible to insert a line if block mem is full !
  if(lineNum < PCIE_REQBEHLINE_NUM && mReqBehLineCount[portHandle][(UInt8)channel] < PCIE_REQBEHLINE_NUM)
  {
    if( CPCIEFpgaFramework::instance().getFpgaTxMemories().insertReqBehLine( portHandle, channel, lineNum,
      mReqBehLineCount[portHandle][( UInt8 )channel] ) )
    {
      mReqBehLineCount[portHandle][(UInt8)channel]++;
      ReqBehLineDefaultSet(portHandle, channel,lineNum);
    }
  }
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::ReqBehLineDelete
(
 const AgtPortHandleT& portHandle, 
 const EPCIEHwChannelFunction& channel, 
 const UInt32& lineNum
 )
{
  if(lineNum < PCIE_REQBEHLINE_NUM && mReqBehLineCount[portHandle][(UInt8)channel] > 0)
  {
    if( CPCIEFpgaFramework::instance().getFpgaTxMemories().removeReqBehLine( portHandle, channel, lineNum,
      mReqBehLineCount[portHandle][( UInt8 )channel] ) )
    {
      mReqBehLineCount[portHandle][(UInt8)channel]--;
      
      ReqBehLineDefaultSet(portHandle, channel,mReqBehLineCount[portHandle][(UInt8)channel]);
    }
  }
}

// ----------------------------------------------------------------------------
UInt32 CPCIEMemoryMgmt::ReqBehLineCountGet
(
  const AgtPortHandleT& portHandle,
  const EPCIEHwChannelFunction& channel
)
{
  return mReqBehLineCount[portHandle][(UInt8)channel];
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::CompBehSet
(
 const AgtPortHandleT& portHandle, 
 const EPCIECompQueue& queue, 
 const UInt32& line, 
 const EPCIECompBeh& prop,
 const UInt32& val
 )
{
  CPCIEFpgaFramework::instance().getFpgaTxMemories().setCompBehProp(portHandle,prop,queue, line, val);

  if(prop == PCIE_COMPBEH_REPEAT)
  {
    updateRepeatFirst_CplBeh( portHandle, queue );
  }

  CompleterRun(portHandle);
}

// ----------------------------------------------------------------------------
UInt32 CPCIEMemoryMgmt::CompBehGet
(
 const AgtPortHandleT& portHandle, 
 const EPCIECompQueue& queue, 
 const UInt32& line, 
 const EPCIECompBeh& prop
 )
{
  return CPCIEFpgaFramework::instance().getFpgaTxMemories().getCompBehProp(portHandle,prop,queue, line);
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::CompBehLineWrite
(
 const AgtPortHandleT& portHandle, 
 const EPCIECompQueue& queue, 
 const UInt32& lineNum, 
 const UInt32& numBytes, 
 const UInt8* mData
 )
{
  LineWrite(portHandle, MEMORYMGMT_LINETYPE_COMPBEH,(UInt8)queue,lineNum,numBytes,mData);

  CompleterRun(portHandle);
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::CompBehLineRead
(
 const AgtPortHandleT& portHandle, 
 const EPCIECompQueue& queue, 
 const UInt32& lineNum, 
 const UInt32& numBytes, 
 UInt8* theData
 )
{
  LineRead(portHandle, MEMORYMGMT_LINETYPE_COMPBEH,(UInt8)queue,lineNum,numBytes,theData);
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::CompBehLineDefaultSet
(
 const AgtPortHandleT& portHandle, 
 const EPCIECompQueue& queue, 
 const UInt32& lineNum
 )
{
  CPCIEFpgaFramework::instance().getFpgaTxMemories().setCompBehLineDefault(portHandle,queue, lineNum);
  
  updateRepeatFirst_CplBeh( portHandle, queue );
  
  CompleterRun(portHandle);
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::CompBehLineDefaultInsert
(
 const AgtPortHandleT& portHandle, 
 const EPCIECompQueue& queue, 
 const UInt32& lineNum
 )
{
  // it's not possible to insert a line if block mem is full !
  if(lineNum < PCIE_COMPBEHLINE_NUM && mCompBehLineCount[portHandle][(UInt8)queue] < PCIE_COMPBEHLINE_NUM)
  {
    if( CPCIEFpgaFramework::instance().getFpgaTxMemories().insertCompBehLine( portHandle, queue, lineNum,
      mCompBehLineCount[portHandle][( UInt8 )queue] ) )
    {
      mCompBehLineCount[portHandle][(UInt8)queue]++;
      CompBehLineDefaultSet(portHandle, queue,lineNum);
    }
  }
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::CompBehLineDelete
(
 const AgtPortHandleT& portHandle, 
 const EPCIECompQueue& queue, 
 const UInt32& lineNum
 )
{
  if(lineNum < PCIE_COMPBEHLINE_NUM && mCompBehLineCount[portHandle][(UInt8)queue] > 0)
  {
    if( CPCIEFpgaFramework::instance().getFpgaTxMemories().removeCompBehLine( portHandle, queue, lineNum,
      mCompBehLineCount[portHandle][( UInt8 )queue] ) )
    {
      mCompBehLineCount[portHandle][(UInt8)queue]--;
      
      CompBehLineDefaultSet(portHandle, queue,mCompBehLineCount[portHandle][(UInt8)queue]);
    }
  }
}

// ----------------------------------------------------------------------------
UInt32 CPCIEMemoryMgmt::CompBehLineCountGet
(
  const AgtPortHandleT& portHandle,
  const EPCIECompQueue& queue
)
{
  return mCompBehLineCount[portHandle][(UInt8)queue];
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::LineWrite
(
 const AgtPortHandleT& portHandle, 
 const CPCIEMemoryMgmt::LineType& lineType,
 const UInt8& resource, 
 const UInt32& lineNum, 
 const UInt32& numBytes, 
 const UInt8* mData
 )
{
  UInt32 numProps = 0x0;
  UInt32 propVal = 0x0;
  UInt32 prop = 0x0;

  bool zeroPropWritten = false;
  
  if( (numBytes % 5) || numBytes == 0)
  {
  }
  else
  {
    numProps = numBytes / 5;
    
    //get interface only once
    for (UInt32 i = 0; i < numProps; i++)
    {
        propVal = 0x0;
        prop = mData[(i*5)+0x0];
        
        for(UInt8 vByte = 1; vByte < 5; vByte++)
        {
          propVal |= mData[(i*5)+vByte] << (8*(vByte-1));
        }
        
      if((prop!=0) || (!zeroPropWritten))
      {
        switch(lineType)
        {
        case MEMORYMGMT_LINETYPE_BLOCK:
          BlockSet(portHandle, (EPCIEHwChannelFunction)resource, lineNum, (EPCIEBlock)prop, propVal);
          break;
        case MEMORYMGMT_LINETYPE_REQBEH:
          ReqBehSet(portHandle,(EPCIEHwChannelFunction)resource, lineNum, (EPCIEReqBeh)prop, propVal);
          break;
        case MEMORYMGMT_LINETYPE_COMPBEH:  
          CompBehSet(portHandle, (EPCIECompQueue)resource, lineNum, (EPCIECompBeh)prop, propVal);
          break;
        default:
          AGT_THROW("CPCIEMemoryMgmt::LineWrite Invalid lineType");
        }
      }

      if (prop == 0)
      {
        zeroPropWritten = true;
      }
    }
  }
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::LineRead
(
 const AgtPortHandleT& portHandle, 
 const CPCIEMemoryMgmt::LineType& lineType,
 const UInt8& resource, 
 const UInt32& lineNum, 
 const UInt32& numBytes, 
 UInt8* theData
 )
{
  UInt32 maxPropNum = 0;
  UInt32 propNum = 0x0;
  UInt32 propVal = 0x0;
  BlockLineType blockLineType = MEMORYMGMT_BLOCKLINETYPE_UNKNOWN;

  if(numBytes != (PCIE_MAX_BLOCKPROP_NUM*5))
  {
    AGT_THROW("CPCIEMemoryMgmt::LineRead error numBytes not correct");
  }

  // get interface to tx memories
  CPCIEFpgaTxMemories& txMemoriesInterface = CPCIEFpgaFramework::instance().getFpgaTxMemories();   

  switch(lineType)
  {
  case MEMORYMGMT_LINETYPE_BLOCK:
    maxPropNum = PCIE_MAX_BLOCKPROP_NUM;
    propNum = PCIE_BLOCK_MARK;

    // find out block line type
    {
      UInt32 fmt = txMemoriesInterface.getBlockProp( portHandle, PCIE_BLOCK_FMT,
        ( EPCIEHwChannelFunction )resource, lineNum );
      UInt32 type = txMemoriesInterface.getBlockProp( portHandle, PCIE_BLOCK_TYPE,
        ( EPCIEHwChannelFunction )resource, lineNum );
      
      // this is mem access
      if(type == 0 || type == 1) 
      {
        blockLineType = (fmt & 0x1 ? MEMORYMGMT_BLOCKLINETYPE_MEM64 : MEMORYMGMT_BLOCKLINETYPE_MEM32);
      }

      if(type == 2)
      {
        blockLineType = MEMORYMGMT_BLOCKLINETYPE_IO;
      }

      if(type == 4 || type == 5)
      {
        blockLineType = MEMORYMGMT_BLOCKLINETYPE_CFG;
      }

      if( ((type >> 3) & 0x2) == 0x2)
      {
        blockLineType = MEMORYMGMT_BLOCKLINETYPE_MSG; 
      }

      if( ((type >> 3) & 0x3) == 0x3)
      {
        blockLineType = ( (fmt >> 1) & 0x1 ? MEMORYMGMT_BLOCKLINETYPE_MSGAS: MEMORYMGMT_BLOCKLINETYPE_MSGASD);
      }
    }
    break;
    
  case MEMORYMGMT_LINETYPE_REQBEH:
    maxPropNum = PCIE_MAX_REQBEHPROP_NUM;
    propNum = PCIE_REQBEH_LEN;    
    break;

  case MEMORYMGMT_LINETYPE_COMPBEH:
    maxPropNum = PCIE_MAX_COMPBEHPROP_NUM;
    propNum = PCIE_COMPBEH_REPEAT;
    break;

  default:
    AGT_THROW("CPCIEMemoryMgmt::LineRead Invalid lineType");
  }
  
  // enter default 'eof'
  for (UInt32 i = 0; i < maxPropNum; i++)
  {
    theData[i*5] = 0xff;
  }
  
  // read properties
  UInt32 currentIndex = 0;
  bool valid = 1;

  for (UInt32 i = currentIndex; i <= propNum; i++)
  {
    switch(lineType)
    {
    case MEMORYMGMT_LINETYPE_BLOCK:
      if(mBlockLineValid[blockLineType].mBlock[i])
      {
        propVal = BlockGet(portHandle, (EPCIEHwChannelFunction)resource, lineNum,(EPCIEBlock)i);
        valid = 1;
      } 
      else
      {
        valid = 0;
      }
      break;

    case MEMORYMGMT_LINETYPE_REQBEH:
      propVal = ReqBehGet(portHandle, (EPCIEHwChannelFunction)resource, lineNum, (EPCIEReqBeh)i);
      break;

    case MEMORYMGMT_LINETYPE_COMPBEH:
      propVal = CompBehGet(portHandle, (EPCIECompQueue)resource, lineNum, (EPCIECompBeh)i);
      break;

    default:
      AGT_THROW("CPCIEMemoryMgmt::LineRead Invalid lineType 2");
    }
    
    if(valid)
    {
      theData[(currentIndex*5)+0] = (UInt8) i;
      theData[(currentIndex*5)+1] = (UInt8) ((propVal>>0) & 0xff);
      theData[(currentIndex*5)+2] = (UInt8) ((propVal>>8) & 0xff);
      theData[(currentIndex*5)+3] = (UInt8) ((propVal>>16) & 0xff);
      theData[(currentIndex*5)+4] = (UInt8) ((propVal>>24) & 0xff);
      currentIndex++;
    }
  } 
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::BlockGenWrite
(
 const AgtPortHandleT& portHandle, 
 const EPCIEHwChannelFunction& channel, 
 const EPCIEBlockGen& prop, 
 const UInt32& val
)
{
  switch(prop)
  {
  case PCIE_BLOCKGEN_STARTLINE:
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setBlockPointers(portHandle,channel,1,val);
    updateRepeatFirst_Block( portHandle, channel );
    break;

  case PCIE_BLOCKGEN_STOPLINE:
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setBlockPointers(portHandle,channel,0,val);
    break;

  case PCIE_BLOCKGEN_REPEAT:
    mSingleShotEnabled[portHandle][(UInt8)channel] = ( val == 0 ? 0 : 1 );
    break;

  case PCIE_BLOCKGEN_ENABLED:
    mBlockGenEnabled[portHandle][(UInt8)channel] = ( val == 0 ? false : true);
    break;

  default:
   AGT_THROW("CPCIEMemoryMgmt::BlockGenWrite Invalid prop");
  }
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::BlockGenDefaultWrite
(
 const AgtPortHandleT& portHandle,
 const EPCIEHwChannelFunction& channel
)
{
  BlockGenWrite(portHandle, channel,PCIE_BLOCKGEN_STARTLINE,0);
  BlockGenWrite(portHandle, channel,PCIE_BLOCKGEN_STOPLINE,0);
  BlockGenWrite(portHandle, channel,PCIE_BLOCKGEN_REPEAT,1);
  BlockGenWrite(portHandle, channel,PCIE_BLOCKGEN_ENABLED,(channel == PCIE_HWCHANNEL_FUNCTION_A ? 1 : 0));
}

// ----------------------------------------------------------------------------
UInt32 CPCIEMemoryMgmt::BlockGenRead
(
 const AgtPortHandleT& portHandle, 
 const EPCIEHwChannelFunction& channel, 
 const EPCIEBlockGen& prop
)
{
  switch(prop)
  {
  case PCIE_BLOCKGEN_STARTLINE:
    return CPCIEFpgaFramework::instance().getFpgaCSPkt().getBlockPointers(portHandle,channel,1);

  case PCIE_BLOCKGEN_STOPLINE:
    return CPCIEFpgaFramework::instance().getFpgaCSPkt().getBlockPointers(portHandle,channel,0);

  case PCIE_BLOCKGEN_REPEAT:   
    return mSingleShotEnabled[portHandle][(UInt8)channel];

  case PCIE_BLOCKGEN_ENABLED:
    return mBlockGenEnabled[portHandle][(UInt8)channel];
    break;

  default:
   AGT_THROW("CPCIEMemoryMgmt::BlockGenWrite Invalid prop");
  }
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::ReqBehGenWrite
(
 const AgtPortHandleT& portHandle, 
 const EPCIEHwChannelFunction& channel, 
 const EPCIEReqBehGen& prop, 
 const UInt32& val
)
{
  switch(prop)
  {
  case PCIE_REQBEHGEN_STARTLINE:
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setReqBehPointers(portHandle,channel,1,val);
    updateRepeatFirst_ReqBeh( portHandle, channel );
    break;

  case PCIE_REQBEHGEN_STOPLINE:
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setReqBehPointers(portHandle,channel,0,val);
    break;

  default:
   AGT_THROW("CPCIEMemoryMgmt::ReqBehGenWrite Invalid prop");
  }
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::ReqBehGenDefaultWrite
(
 const AgtPortHandleT& portHandle, 
 const EPCIEHwChannelFunction& channel
)
{
  ReqBehGenWrite(portHandle, channel,PCIE_REQBEHGEN_STARTLINE,0);
  ReqBehGenWrite(portHandle, channel,PCIE_REQBEHGEN_STOPLINE,0);
 }

// ----------------------------------------------------------------------------
UInt32 CPCIEMemoryMgmt::ReqBehGenRead
(
 const AgtPortHandleT& portHandle, 
 const EPCIEHwChannelFunction& channel, 
 const EPCIEReqBehGen& prop
)
{
  switch(prop)
  {
  case PCIE_REQBEHGEN_STARTLINE:
    return CPCIEFpgaFramework::instance().getFpgaCSPkt().getReqBehPointers(portHandle,channel,1);

  case PCIE_REQBEHGEN_STOPLINE:
    return CPCIEFpgaFramework::instance().getFpgaCSPkt().getReqBehPointers(portHandle,channel,0);

  default:
   AGT_THROW("CPCIEMemoryMgmt::ReqBehGenWrite Invalid prop");
  }
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::CompBehGenWrite
(
 const AgtPortHandleT& portHandle, 
 const EPCIECompQueue& queue, 
 const EPCIECompBehGen& prop, 
 const UInt32& val
)
{
  switch(prop)
  {
  case PCIE_COMPBEHGEN_STARTLINE:
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setCompBehPointers(portHandle,queue,1,val);
    updateRepeatFirst_CplBeh( portHandle, queue );
    break;

  case PCIE_COMPBEHGEN_STOPLINE:
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setCompBehPointers(portHandle,queue,0,val);
    break;

  default:
   AGT_THROW("CPCIEMemoryMgmt::CompBehGenWrite Invalid prop");
  }

  CompleterRun(portHandle);
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::CompBehGenDefaultWrite
(
 const AgtPortHandleT& portHandle, 
 const EPCIECompQueue& queue
)
{
  CompBehGenWrite(portHandle, queue,PCIE_COMPBEHGEN_STARTLINE,0);
  CompBehGenWrite(portHandle, queue,PCIE_COMPBEHGEN_STOPLINE,0);
}

// ----------------------------------------------------------------------------
UInt32 CPCIEMemoryMgmt::CompBehGenRead
(
 const AgtPortHandleT& portHandle, 
 const EPCIECompQueue& queue, 
 const EPCIECompBehGen& prop
)
{
  switch(prop)
  {
  case PCIE_COMPBEHGEN_STARTLINE:
    return CPCIEFpgaFramework::instance().getFpgaCSPkt().getCompBehPointers(portHandle,queue,1);

  case PCIE_COMPBEHGEN_STOPLINE:
    return CPCIEFpgaFramework::instance().getFpgaCSPkt().getCompBehPointers(portHandle,queue,0);

  default:
   AGT_THROW("CPCIEMemoryMgmt::CompBehGenWrite Invalid prop");
  }
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::CompleterRun
(
  const AgtPortHandleT& portHandle
)
{
  CPCIEFpgaFramework::instance().CompleterRun(portHandle);
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::RequesterRun
(
  const AgtPortHandleT& portHandle
)
{
  bool valid = 1;

  // Check bus master enable bit
  //UInt32 tmp = CPCIEFpgaFramework::instance().getConfigSpace().ConfigSpaceDirectRead( portHandle, CPCIEConfigSpace::CONFIGSPACEENTRYOFFSET_DW_STATUS_COMMAND * 4);

  // Will enable later
  /*if( ( ( tmp >> 2 ) & 0x1 ) == 0 )
  {
  valid = 0;
  }*/

  if( valid )
  {
    UInt32 startLine;
    UInt32 stopLine;

    for( UInt8 channelNum = 0; channelNum < PCIEMEMORYMGMT_NOFHWCHANNELS; channelNum++ )
    {
      if( mBlockGenEnabled[portHandle][channelNum] )
      {
        startLine = BlockGenRead( portHandle, ( EPCIEHwChannelFunction )channelNum, PCIE_BLOCKGEN_STARTLINE );
        stopLine = BlockGenRead( portHandle, ( EPCIEHwChannelFunction )channelNum, PCIE_BLOCKGEN_STOPLINE );

        for( UInt32 lineNum = startLine; lineNum < stopLine + 1; lineNum++ )
        {
          CPCIEFpgaFramework::instance().getFpgaTxMemories().update64BitSettings( portHandle,
            ( EPCIEHwChannelFunction )channelNum, lineNum );
        }
      }
    }

    UInt32 mode = CPCIEFpgaFramework::instance().getFpgaCSMis().getExerciserMode( portHandle );
    UInt32 runMask = CPCIEFpgaFramework::instance().getFpgaCSPkt().getTxCtrl( portHandle );
    UInt32 runMaskExtended = CPCIEFpgaFramework::instance().getFpgaCSPkt().getTxCtrlExtended( portHandle );

    /* mask for all the channels which are enabled */
    /* mask for all the channels for all single shot mode */
    if( mBlockGenEnabled[portHandle][( UInt8 ) PCIE_HWCHANNEL_FUNCTION_A] )
    {
      runMask |= 1 << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_A_BIT;
      runMask |= mSingleShotEnabled[portHandle][( UInt8 ) PCIE_HWCHANNEL_FUNCTION_A]
      << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_A_SINGLESHOT_MODE_BIT;
    }
    if( mBlockGenEnabled[portHandle][( UInt8 )PCIE_HWCHANNEL_FUNCTION_B] )
    {
      runMask |= 1 << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_B_BIT;
      runMask |= mSingleShotEnabled[portHandle][( UInt8 )PCIE_HWCHANNEL_FUNCTION_B]
      << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_B_SINGLESHOT_MODE_BIT;
    }
    if( mBlockGenEnabled[portHandle][( UInt8 )PCIE_HWCHANNEL_FUNCTION_C] )
    {
      runMask |= 1 << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_C_BIT;
      runMask |= mSingleShotEnabled[portHandle][( UInt8 )PCIE_HWCHANNEL_FUNCTION_C]
      << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_C_SINGLESHOT_MODE_BIT;
    }
    if( mBlockGenEnabled[portHandle][( UInt8 ) PCIE_HWCHANNEL_FUNCTION_BVF1] )
    {
      runMask |= 1 << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_BVF1_BIT;
      runMask |= mSingleShotEnabled[portHandle][( UInt8 )PCIE_HWCHANNEL_FUNCTION_BVF1]
      << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_BVF1_SINGLESHOT_MODE_BIT;
    }
    if( mBlockGenEnabled[portHandle][( UInt8 )PCIE_HWCHANNEL_FUNCTION_BVF2] )
    {
      runMask |= 1 << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_BVF2_BIT;
      runMask |= mSingleShotEnabled[portHandle][( UInt8 )PCIE_HWCHANNEL_FUNCTION_BVF2]
      << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_BVF2_SINGLESHOT_MODE_BIT;
    }
    if( mBlockGenEnabled[portHandle][( UInt8 ) PCIE_HWCHANNEL_FUNCTION_CVF1] )
    {
      runMask |= 1 << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_CVF1_BIT;
      runMask |= mSingleShotEnabled[portHandle][( UInt8 ) PCIE_HWCHANNEL_FUNCTION_CVF1]
      << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_CVF1_SINGLESHOT_MODE_BIT;
    }
    if( mBlockGenEnabled[portHandle][( UInt8 ) PCIE_HWCHANNEL_FUNCTION_CVF2] )
    {
      runMask |= 1 << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_CVF2_BIT;
      runMask |= mSingleShotEnabled[portHandle][( UInt8 ) PCIE_HWCHANNEL_FUNCTION_CVF2]
      << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_CVF2_SINGLESHOT_MODE_BIT;
    }

    if( mBlockGenEnabled[portHandle][( UInt8 )PCIE_HWCHANNEL_FUNCTION_D] )
    {
      runMaskExtended |= 1 << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_D_BIT;
      runMaskExtended |= mSingleShotEnabled[portHandle][( UInt8 )PCIE_HWCHANNEL_FUNCTION_D]
      << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_D_SINGLESHOT_MODE_BIT;
    }
    if( mBlockGenEnabled[portHandle][( UInt8 )PCIE_HWCHANNEL_FUNCTION_E] )
    {
      runMaskExtended |= 1 << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_E_BIT;
      runMaskExtended |= mSingleShotEnabled[portHandle][( UInt8 )PCIE_HWCHANNEL_FUNCTION_E]
      << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_E_SINGLESHOT_MODE_BIT;
    }
    if( mBlockGenEnabled[portHandle][( UInt8 ) PCIE_HWCHANNEL_FUNCTION_DVF1] )
    {
      runMaskExtended |= 1 << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_DVF1_BIT;
      runMaskExtended |= mSingleShotEnabled[portHandle][( UInt8 )PCIE_HWCHANNEL_FUNCTION_DVF1]
      << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_DVF1_SINGLESHOT_MODE_BIT;
    }
    if( mBlockGenEnabled[portHandle][( UInt8 )PCIE_HWCHANNEL_FUNCTION_DVF2] )
    {
      runMaskExtended |= 1 << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_DVF2_BIT;
      runMaskExtended |= mSingleShotEnabled[portHandle][( UInt8 )PCIE_HWCHANNEL_FUNCTION_DVF2]
      << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_DVF2_SINGLESHOT_MODE_BIT;
    }
    if( mBlockGenEnabled[portHandle][( UInt8 ) PCIE_HWCHANNEL_FUNCTION_EVF1] )
    {
      runMaskExtended |= 1 << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_EVF1_BIT;
      runMaskExtended |= mSingleShotEnabled[portHandle][( UInt8 ) PCIE_HWCHANNEL_FUNCTION_EVF1]
      << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_EVF1_SINGLESHOT_MODE_BIT;
    }
    if( mBlockGenEnabled[portHandle][( UInt8 ) PCIE_HWCHANNEL_FUNCTION_EVF2] )
    {
      runMaskExtended |= 1 << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_EVF2_BIT;
      runMaskExtended |= mSingleShotEnabled[portHandle][( UInt8 ) PCIE_HWCHANNEL_FUNCTION_EVF2]
      << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_EVF2_SINGLESHOT_MODE_BIT;
    }

     if( ( EExerciserMode )mode == EFPGA_PCIE_MODE )
    {
      runMask &= ~( 1 << CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_BVF1_BIT );
      runMask &= ~( 1 << CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_BVF2_BIT );
      runMask &= ~( 1 << CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_CVF1_BIT );
      runMask &= ~( 1 << CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_CVF2_BIT );
      runMask &= ~( 1 << CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_BVF1_SINGLESHOT_MODE_BIT );
      runMask &= ~( 1 << CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_BVF2_SINGLESHOT_MODE_BIT );
      runMask &= ~( 1 << CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_CVF1_SINGLESHOT_MODE_BIT );
      runMask &= ~( 1 << CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_CVF2_SINGLESHOT_MODE_BIT );

      runMaskExtended &= ~( 1 << CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_DVF1_BIT );
      runMaskExtended &= ~( 1 << CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_DVF2_BIT );
      runMaskExtended &= ~( 1 << CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_EVF1_BIT );
      runMaskExtended &= ~( 1 << CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_EVF2_BIT );
      runMaskExtended &= ~( 1 << CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_DVF1_SINGLESHOT_MODE_BIT );
      runMaskExtended &= ~( 1 << CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_DVF2_SINGLESHOT_MODE_BIT );
      runMaskExtended &= ~( 1 << CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_EVF1_SINGLESHOT_MODE_BIT );
      runMaskExtended &= ~( 1 << CPCIEFpgaCSPkt::FPGATXMISC_CTRL_REQ_GEN_EVF2_SINGLESHOT_MODE_BIT );
    }

    // Start only the requesters by enabling the FPGATXMISC_CTRL_START_REQ_GEN_BIT bit 
    runMask |= 1 << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_START_REQ_GEN_BIT;

    CPCIEFpgaFramework::instance().getFpgaCSPkt().setTxCtrlExtended( portHandle, runMaskExtended );
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setTxCtrl( portHandle, runMask );
  }
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::RequesterStop
(
  const AgtPortHandleT& portHandle
)
{
  UInt32 stopMask = CPCIEFpgaFramework::instance().getFpgaCSPkt().getTxCtrl( portHandle );

  // Stop the exerciser
  stopMask |= 0x1 << ( UInt8 )CPCIEFpgaCSPkt::FPGATXMISC_CTRL_STOP_REQ_GEN_BIT;

  CPCIEFpgaFramework::instance().getFpgaCSPkt().setTxCtrl( portHandle, stopMask );
}

// ----------------------------------------------------------------------------
UInt32 CPCIEMemoryMgmt::GetHWChannelState
(
  const AgtPortHandleT& portHandle,
  const EPCIEHwChannelFunction& channel
)
{
  CPCIEFpgaCSPkt::TxMiscBlockMemoryState blockMemoryState;

  blockMemoryState = CPCIEFpgaFramework::instance().getFpgaCSPkt().getBlockMemoryState( portHandle, channel );

  switch( blockMemoryState )
  {
    case CPCIEFpgaCSPkt::FPGATXMISC_BLOCKMEMORY_STATE_INVALID:
      return (UInt32) PCIE_HWCHANNELSTATEV_INVALID;
    case CPCIEFpgaCSPkt::FPGATXMISC_BLOCKMEMORY_STATE_STOPPING:
      return (UInt32) PCIE_HWCHANNELSTATEV_STOPPING;
    case CPCIEFpgaCSPkt::FPGATXMISC_BLOCKMEMORY_STATE_RUNNING:
      return (UInt32) PCIE_HWCHANNELSTATEV_RUNNING;
    case CPCIEFpgaCSPkt::FPGATXMISC_BLOCKMEMORY_STATE_STOPPED:
      return (UInt32) PCIE_HWCHANNELSTATEV_STOPPED;
    default:
      return (UInt32) PCIE_HWCHANNELSTATEV_INVALID;
  }
}

// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::updateRepeatFirst_Block
(
  const AgtPortHandleT& portHandle,
  const EPCIEHwChannelFunction& channel
)
{
  UInt32 val = 0x0;
  UInt32 startLine = 0x0;

  startLine = BlockGenRead( portHandle, channel, PCIE_BLOCKGEN_STARTLINE );
  val = BlockGet( portHandle, channel, startLine, PCIE_BLOCK_REPEAT );

  /* Set the Block Memory Repeat bits for Channel A, B*/
  if( ( PCIE_HWCHANNEL_FUNCTION_A == channel ) ||
      ( PCIE_HWCHANNEL_FUNCTION_B == channel ) )
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setBlockRepeatFirst( portHandle, channel, val );

  /* Set the Block Memory Repeat bits for Channel C, BVF1 Register Map :cspkt:0x0008C*/
  if( ( PCIE_HWCHANNEL_FUNCTION_C == channel ) ||
      ( PCIE_HWCHANNEL_FUNCTION_BVF1 == channel ) )
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setBlockRepeatSecond( portHandle, channel, val );

  /* Set the Block Memory Repeat bits for Channel BVF2,CVF1,CVF2*/
  if( ( PCIE_HWCHANNEL_FUNCTION_BVF2 == channel ) ||
      ( PCIE_HWCHANNEL_FUNCTION_CVF1 == channel ) ||
      ( PCIE_HWCHANNEL_FUNCTION_CVF2 == channel ) )
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setBlockRepeatThird( portHandle, channel, val );

  if( ( PCIE_HWCHANNEL_FUNCTION_D == channel ) ||
      ( PCIE_HWCHANNEL_FUNCTION_E == channel ) ||
      ( PCIE_HWCHANNEL_FUNCTION_DVF1 == channel ) ||
      ( PCIE_HWCHANNEL_FUNCTION_DVF2 == channel ) )
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setBlockRepeatFourth( portHandle, channel, val );

  if( ( PCIE_HWCHANNEL_FUNCTION_EVF1 == channel ) ||
      ( PCIE_HWCHANNEL_FUNCTION_EVF2 == channel ) )
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setBlockRepeatFifth( portHandle, channel, val );
}
  
// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::updateRepeatFirst_ReqBeh
(
  const AgtPortHandleT& portHandle,
  const EPCIEHwChannelFunction& channel
)
{
  UInt32 val = 0x0;
  UInt32 startLine = 0x0;

  startLine = ReqBehGenRead( portHandle, channel, PCIE_REQBEHGEN_STARTLINE );
  val = ReqBehGet( portHandle, channel, startLine, PCIE_REQBEH_REPEAT );

  /* Set the Requester Behavior memory bits for Channel A, B*/
  if( ( PCIE_HWCHANNEL_FUNCTION_A == channel ) ||
      ( PCIE_HWCHANNEL_FUNCTION_B == channel ) )
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setReqBehRepeatFirst( portHandle, channel, val );

  /* Set the Requester Behavior memory bits for Channel C, BVF1 Register Map : cspkt : 0x00094 */
  if( ( PCIE_HWCHANNEL_FUNCTION_C == channel ) ||
      ( PCIE_HWCHANNEL_FUNCTION_BVF1 == channel ) )
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setReqBehRepeatSecond( portHandle, channel, val );

  /* Set the Requester Behavior memory bits for Channel BVF2,CVF1,CVF2. */
  if( ( PCIE_HWCHANNEL_FUNCTION_BVF2 == channel ) ||
      ( PCIE_HWCHANNEL_FUNCTION_CVF1 == channel ) ||
      ( PCIE_HWCHANNEL_FUNCTION_CVF2 == channel ) )
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setReqBehRepeatThird( portHandle, channel, val );

  if( ( PCIE_HWCHANNEL_FUNCTION_D == channel ) ||
      ( PCIE_HWCHANNEL_FUNCTION_E == channel ) ||
      ( PCIE_HWCHANNEL_FUNCTION_DVF1 == channel ) ||
      ( PCIE_HWCHANNEL_FUNCTION_DVF2 == channel ) )
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setReqBehRepeatFourth( portHandle, channel, val );

  if( ( PCIE_HWCHANNEL_FUNCTION_EVF1 == channel ) ||
      ( PCIE_HWCHANNEL_FUNCTION_EVF2 == channel ) )
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setReqBehRepeatFifth( portHandle, channel, val );
}
  
// ----------------------------------------------------------------------------
void CPCIEMemoryMgmt::updateRepeatFirst_CplBeh
(
  const AgtPortHandleT& portHandle,
  const EPCIECompQueue& queue
)
  {
  UInt32 val = 0x0;
  UInt32 startLine = 0x0;

  startLine = CompBehGenRead( portHandle, queue, PCIE_COMPBEHGEN_STARTLINE );
  val = CompBehGet( portHandle, queue, startLine, PCIE_COMPBEH_REPEAT );

  if( ( PCIE_COMPQUEUE_0 == queue ) ||
      ( PCIE_COMPQUEUE_1 == queue ) )
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setCompBehRepeatFirst( portHandle, queue, val );

  if( ( PCIE_COMPQUEUE_2 == queue ) ||
      ( PCIE_COMPQUEUE_3 == queue ) ||
      ( PCIE_COMPQUEUE_4 == queue ) )
    CPCIEFpgaFramework::instance().getFpgaCSPkt().setCompBehRepeatSecond( portHandle, queue, val );
}